#ifndef DMATRIX_H
#define DMATRIX_H

//#define DEBUG

//#include "matrix.h"
#include "tools.h"
#include "mpi_omp.h"
#include "diterator.h"
#include<math.h>

//!  template Distributed Matrix Class
/*!
  This template class defines the dstributed matrix
  T = type of data in the matrix
*/
//-------------------------------------------------------------------------------
template<class T> class DMatrix
//-------------------------------------------------------------------------------
{
public:
    //! typedef for the iterator
    /*! typedef for the iterator on the distributed matrix */
    typedef DIterator<T> iterator;
    typedef T * neighbors;

protected:
    //! typedef for the iterator
    /*! typedef for the iterator on the distributed matrix to save the cuurent rank of the iterator in the current submatrice*/
    //typedef unsigned int iterator_rank;

    //=======================OMP/MPI ATTRIBUTES=======================//
    //! my OMP thread
    int omp_rank;
    //! initialized by Mpiomp class
    int mpi_rank,mpi_nb,omp_nb;

    //! initialized if the matrix is local with no MPI or OMP division
    bool local;

    //! number for shared memory access of other threads dmatrix
    int shared_mem;

    //! column and row position in the openMP division of the matrix
    //in order to know if our neighbors are OMP or MPI
    int loc_col,loc_row;
    //! total number of columns and rows in the openMP division of the matrix
    //in order to know if our neighbors are OMP or MPI
    int loc_cols,loc_rows;
    //! column and row position in the general division
    int col,row;
    //! total number of columns and rows for he current decomposition (MPI or OpenMP)
    int cols,rows;
    //! decalage number in order to read values in file
    //! MPI border to exchange
    int border;
    //===========matrix with border
    //border values for MPI exchanges
    /*T * upValues;
    T * downValues;
    T * leftValues;
    T * rightValues;
    T * uplValues;
    T * uprValues;
    T * downlValues;
    T * downrValues;*/
    //the list of shared matrices of the other OMP threads
    //Matrix<T> ** list;

    //! cache optimization, precedent 8 neighbors
    T * precedent8N;
    //! cache optimization, precedent 8 neighbors's rank
    unsigned int precedent8NRank;

    //T * result;
    //=============================================================//

    //=======================DATA ATTRIBUTES=======================//
    //! The data !
    /*!
      in every case (hybrid or not) we will need a matrix containing the data
    */
    //=============no matrix.h
    //Matrix<T> * subMatrix;
    T * data;
    //! Header defining the current global matrix
    HEADER head;
    //! Header defining the current local matrix
    HEADER loc_head;
    //! Header for local matrix with the border
    HEADER border_head;
    //! Reading offset
    unsigned int offset_decomp;
    unsigned int offset_border;
    //=============================================================//


    //=======================MATRICE CREATION METHODS=======================//
    //! Read the header of the file
    /*!
      We are reading some binary MNTs, this method permits to read the header of the binary file
      \param binFile is the path to the file
    */
    //-------------------------------------------------------------------------------
    void readHeader(const char * binFile)
    //-------------------------------------------------------------------------------
    {
        std::ifstream f(binFile, std::ios::binary | std::ios::in);
        f.read(reinterpret_cast<char*>(&head),sizeof(HEADER));

        /*int w = floor(head.width/20);
        head.width=w;
        int h = floor(head.height/20);
        head.height=h;*/

    }

    //-------------------------------------------------------------------------------
    /*HEADER read(char * binFile)
    //-------------------------------------------------------------------------------
    {
      std::stringstream st;

      std::ifstream f(binFile, std::ios::binary | std::ios::in);
      f.seekg(sizeof(HEADER),std::ios::beg);
    for(int i=0;i<head.width*head.height;i++)
    {
    float val;
    f.read(reinterpret_cast<char*>(&val),sizeof(T));
    if(i%head.width==0)
    st<<"\n";
    st<<val<<" ";
    }

    Mpiomp::print(st.str());
    }*/
    //-------------------------------------------------------------------------------

    //=======================INIT METHODS=======================//
    //! init the distributed matrix
    /*!
      called by each constructor
      \param local bolean who says if the matrix will or won't be local
    */
    //-------------------------------------------------------------------------------
    void init(bool loc, int b)
    //-------------------------------------------------------------------------------
    {
        //===========matrix with border
        /*upValues=NULL;
        downValues=NULL;
        leftValues=NULL;
        rightValues=NULL;
        uplValues=NULL;
        uprValues=NULL;
        downlValues=NULL;
        downrValues=NULL;*/
        //list=NULL;

        precedent8N = new T[8];
        precedent8NRank = -555;

        border=b;
        offset_border = 2*border;

        local=loc;
        if (local) {
            mpi_rank=0;
            mpi_nb=1;
        } else {
            mpi_rank = Mpiomp::mpi_rank;
            mpi_nb = Mpiomp::mpi_nb;
        }
#ifdef _OPENMP
        omp_nb = Mpiomp::omp_nb;
        omp_rank = omp_get_thread_num();
#else
        omp_nb = 0;
        omp_rank = -1;
#endif
    }
    //-------------------------------------------------------------------------------

    //! Init the values of the matrix
    /*!
      fill with -1
    */
    //-------------------------------------------------------------------------------
    void initValues(int value)
    //-------------------------------------------------------------------------------
    {
        for(unsigned int i=0; i<border_head.width*border_head.height; i++) {
            data[i]=value;
        }
        //===========matrix with border
        /*for(unsigned int i=0;i<loc_head.width*loc_head.height;i++)
          {
        data[i]=value;
          }*
          //=============no matrix.h
          /*for(unsigned int j=0;j<loc_head.height;j++)
          {
        for(unsigned int k=0;k<loc_head.width;k++)
          {
            unsigned int rank = j*loc_head.width + k;
            subMatrix->fillMatrix(value,rank);
          }
          }*/
    }
    //-------------------------------------------------------------------------------
    //===========================================================//

    //=======================MATRICE CREATION METHODS=======================//

    //! Read the zone file concerned for the current MPI process
    /*!
      read the MPI prcess zone, and spread the data in the subMatrices OpenMP
      only one submatrice if OpenMP is disabled
      \param binFile is binary file to read
    */
    //-------------------------------------------------------------------------------
    void readZone(const char * binFile)
    //-------------------------------------------------------------------------------
    {
        //read the zone of the file for my own block of data
        std::ifstream f(binFile, std::ios::binary | std::ios::in);

        //move to the begining of the bloc of data concerned
        //begining of the bloc : globWidth*machHeight*(mach_row-1) + (mach_col-1)*machWidth
        unsigned int mvline;
        mvline = sizeof(HEADER) + (head.width*loc_head.height*row+col*loc_head.width)*sizeof(T);

        f.seekg(mvline,std::ios::beg);

        //read by lines
        for(unsigned int j=0; j<loc_head.height; j++) {
            if(j>0) {
                //move to the next line of the bloc
                unsigned int mv;
                if(col!=(cols-1))
                    mv = ((cols-1) * loc_head.width)-offset_decomp;
                else
                    mv = ((cols-1) * loc_head.width);
                f.seekg(mv*sizeof(T),std::ios::cur);
            }

            for(unsigned int k=0; k<loc_head.width; k++) {
                T val;

                //if we are in the uncomplete zone of the last column or row we go past the global width or height
                //in this case we fill with nodata value
                unsigned int curX = col*loc_head.width+k;
                unsigned int curY = row*loc_head.height+j;

                //===========matrix with border
                //unsigned int rank = j*loc_head.width + k;
                unsigned int rank = border*border_head.width + j*border_head.width + k + border;

                if((curX<=head.width-1) && (curY<=head.height-1)) {
                    f.read(reinterpret_cast<char*>(&val),sizeof(T));
                    data[rank]=val;
                    //=============no matrix.h
                    //subMatrix->fillMatrix(val,rank);
                } else
                    data[rank]=head.nodata;
                //=============no matrix.h
                //subMatrix->fillMatrix(head.nodata,rank);
            }
        }
    }
    //-------------------------------------------------------------------------------

    //! create subMatrice
    /*!
      \param binFile is the path to the binary file of data
      \param read precise if we have a file to read or not
    */
    //-------------------------------------------------------------------------------
    void createSubMatrice(const char * binFile, bool read, int value=-1)
    //-------------------------------------------------------------------------------
    {
        //block division with the total number of MPI+OMP processes
        //(we have to suppose that the cluster is homogenous in core by node, otherwise exhanges will be necessaries
        if(local==false) {
            int nbBlocs;
#ifdef _OPENMP
            nbBlocs = mpi_nb*omp_nb;
#else
            nbBlocs = mpi_nb;
#endif
            //division of the matrix in cols and rows
            cols = floor(sqrt(nbBlocs));
            int nb = 1;
            while(nbBlocs%cols!=0) {
                cols = floor(sqrt(nbBlocs))-nb;
                nb++;
            }
            rows = nbBlocs/cols;

#ifdef DEBUG
            std::stringstream st;
            st<<"rows and cols :"<<rows<<","<<cols<<"\n";
            Mpiomp::print(st.str());
#endif
#ifdef _OPENMP

            //block division for the OMP thread to get the number of blocks for each machine
            loc_cols = floor(sqrt(omp_nb));
            int nb2 = 1;
            while(omp_nb%loc_cols!=0) {
                loc_cols = floor(sqrt(omp_nb))-nb2;
                nb2++;
            }
            loc_rows = omp_nb/loc_cols;
            loc_row=floor(omp_rank/loc_cols);
            loc_col=omp_rank%loc_cols;

#ifdef DEBUG
            std::stringstream st2;
            st2<<"OMP locrow and loccol :"<<loc_row<<","<<loc_col<<"\n";
            st2<<"OMP locrows and loccols :"<<loc_rows<<","<<loc_cols<<"\n";
            Mpiomp::print(st2.str());
#endif

            //col and row in the global division
            col = (loc_col + mpi_rank * loc_cols)%cols;
            int myMpiLine = mpi_rank/(cols/loc_cols);
            row = (myMpiLine*loc_rows)+loc_row;

#ifdef DEBUG
            std::stringstream st3;
            st3<<"OMP row and col :"<<row<<","<<col<<"\n";
            Mpiomp::print(st3.str());
#endif
#else
            row=floor(mpi_rank/cols);
            col=mpi_rank%cols;

#ifdef DEBUG
            std::stringstream st3;
            st3<<"MPI row and col :"<<row<<","<<col<<"\n"<<"\n";
            Mpiomp::print(st3.str());
#endif
#endif
            //fill my local header
            offset_decomp=0;
            //if the width of the matrix can be divided by the number of cols
            if(head.width%cols==0)
                loc_head.width=floor(head.width/cols);
            //else we add one to the width of the submatrix and the last column will not be complete
            //in addition to this we have to take into account an offset to read values with the last uncomplete column
            else {
                loc_head.width=floor(head.width/cols)+1;
                offset_decomp = loc_head.width*cols - head.width;
            }
            //if the height of the matrix can be divided by the number of rows
            if(head.height%rows==0)
                loc_head.height=floor(head.height/rows);
            //else we add one to the height of the submatrix and the last row will not be complete
            else
                loc_head.height=floor(head.height/rows)+1;

            loc_head.x=head.x+col*loc_head.width*head.spacing;
            loc_head.y=head.y-row*loc_head.height*head.spacing;
            loc_head.spacing=head.spacing;
            loc_head.nodata=head.nodata;
        } else {
            loc_head.x = head.x;
            loc_head.y = head.y;
            loc_head.width = head.width;
            loc_head.height = head.height;
            loc_head.spacing = head.spacing;
            loc_head.nodata = head.nodata;

            col=0;
            row=0;
            loc_col=0;
            loc_row=0;
            cols=1;
            rows=1;
            loc_cols=1;
            loc_rows=1;

#ifdef DEBUG
            std::stringstream stg;
            stg<<"Size of the Head blocks w,h:"<<head.width<<", "<<head.height<<"\n";
            stg<<"Size of the blocks w,h:"<<loc_head.width<<", "<<loc_head.height<<"\n"<<"\n";
            Mpiomp::print(stg.str());
#endif
        }

        //===========matrix with border
        border_head.x = loc_head.x - border*loc_head.spacing;
        border_head.y = loc_head.y - border*loc_head.spacing;
        border_head.width = loc_head.width + offset_border;
        border_head.height = loc_head.height + offset_border;

        //creation of the matrix
        //=============no matrix.h
        //subMatrix = new Matrix<T>(loc_head,col,row);
        //===========matrix with border
        //data = new T[loc_head.width*loc_head.height];
        data = new T[border_head.width*border_head.height];

        //call the readzone method to read in file
        if(read) {
            initValues(value);
            readZone(binFile);
        } else
            initValues(value);

    }
    //-------------------------------------------------------------------------------

    //===================================================================================//

    //=======================GET MPI BORDERS METHODS=======================//
    //! get rank from a column number and a row number
    /*!
      \param column is the column number
      \param columns is the total number of columns
      \param row is the row number
      \return the rank
    */
    //-------------------------------------------------------------------------------
    unsigned int getRank(int column, int columns, int row)
    //-------------------------------------------------------------------------------
    {
        unsigned int res = row*columns+column;
        return res;
    }
    //-------------------------------------------------------------------------------


    //! get the mpi rank from the absolute rank with all MPI and omp processes
    /*!
      \param absRank is the absolute rank
      \return the mpi rank
    */
    //-------------------------------------------------------------------------------
    int getMPIRank(int absRank)
    //-------------------------------------------------------------------------------
    {
        int mpiCols = cols/loc_cols;
        int mpiCol = (absRank/loc_cols)%mpiCols;
        int mpiRow = absRank/(cols*loc_rows);
        return getRank(mpiCol,mpiCols,mpiRow);
    }
    //-------------------------------------------------------------------------------

    //! get the mpi rank from the absolute rank with all MPI and omp processes
    /*!
      \param absRank is the absolute rank
      \return the mpi rank
    */
    //-------------------------------------------------------------------------------
    int getOMPRank(int absRank)
    //-------------------------------------------------------------------------------
    {
        int locCol = (absRank%cols)%loc_cols;
        int locRow = (absRank/cols)%loc_rows;
        return getRank(locCol,loc_cols,locRow);
    }
    //-------------------------------------------------------------------------------

    //! get ranks
    /*!
      Get the ranks of the 8 neighbors for MPI exchanges
      \param pointer on a table of ranks (0 to 7)
      \return
      if the value is negative then the neighbour is an OMP neighbour, no MPI exchange is needed
      if the value is -55555 then there is no neighbor in this direction
      if the value is >0 it represent the MPI rank concerned by the exchange
    */
    //-------------------------------------------------------------------------------
    void get8Ranks(int * mpi_ranks, int * omp_ranks)
    //-------------------------------------------------------------------------------
    {
        //up
#ifdef _OPENMP
        if(loc_row-1>=0) {
            mpi_ranks[1] = mpi_rank;
            omp_ranks[1] = getRank(loc_col,loc_cols,loc_row-1);
        }
        //mpi exchanges needed
        else if(loc_row-1<0 && row-1>=0) {
            unsigned int absRank = getRank(col,cols,row-1);
            mpi_ranks[1] = getMPIRank(absRank);
            omp_ranks[1] = getOMPRank(absRank);
#else
        if(row-1>=0) {
            mpi_ranks[1] = getRank(col,cols,row-1);
#endif
        } else {
            mpi_ranks[1] = -55555;
#ifdef _OPENMP
            omp_ranks[1] = -55555;
#endif
        }

        //down
#ifdef _OPENMP
        if(loc_row+1<loc_rows) {
            mpi_ranks[5] = mpi_rank;
            omp_ranks[5] = getRank(loc_col,loc_cols,loc_row+1);
        }
        //mpi exchanges needed
        else if(loc_row+1>=loc_rows && row+1<rows) {
            unsigned int absRank = getRank(col,cols,row+1);
            mpi_ranks[5] = getMPIRank(absRank);
            omp_ranks[5] = getOMPRank(absRank);
#else
        if(row+1<rows) {
            mpi_ranks[5] = getRank(col,cols,row+1);
#endif
        } else {
            mpi_ranks[5] = -55555;
#ifdef _OPENMP
            omp_ranks[5] = -55555;
#endif
        }

        //right
#ifdef _OPENMP
        if(loc_col+1<loc_cols) {
            mpi_ranks[3] = mpi_rank;
            omp_ranks[3] = getRank(loc_col+1,loc_cols,loc_row);
        }
        //mpi exchanges needed
        else if(loc_col+1>=loc_cols && col+1<cols) {
            unsigned int absRank = getRank(col+1,cols,row);
            mpi_ranks[3] = getMPIRank(absRank);
            omp_ranks[3] = getOMPRank(absRank);
#else
        if(col+1<cols) {
            mpi_ranks[3] = getRank(col+1,cols,row);
#endif
        } else {
            mpi_ranks[3] = -55555;
#ifdef _OPENMP
            omp_ranks[3] = -55555;
#endif
        }

        //left
#ifdef _OPENMP
        if(loc_col-1>=0) {
            mpi_ranks[7] = mpi_rank;
            omp_ranks[7] = getRank(loc_col-1,loc_cols,loc_row);
        }
        //mpi exchanges needed
        else if(loc_col-1<0 && col-1>=0) {
            unsigned int absRank = getRank(col-1,cols,row);
            mpi_ranks[7] = getMPIRank(absRank);
            omp_ranks[7] = getOMPRank(absRank);
#else
        if(col-1>=0) {
            mpi_ranks[7] = getRank(col-1,cols,row);
#endif
        } else {
            mpi_ranks[7] = -55555;
#ifdef _OPENMP
            omp_ranks[7] = -55555;
#endif
        }

        //up left
#ifdef _OPENMP
        if(loc_col-1>=0 && loc_row-1>=0) {
            mpi_ranks[0] = mpi_rank;
            omp_ranks[0] = getRank(loc_col-1,loc_cols,loc_row-1);
        } else if((loc_col-1<0 || loc_row-1<0) && (col-1>=0 && row-1>=0)) {
            unsigned int absRank = getRank(col-1,cols,row-1);
            mpi_ranks[0] = getMPIRank(absRank);
            omp_ranks[0] = getOMPRank(absRank);
#else
        if(col-1>=0 && row-1>=0) {
            mpi_ranks[0] = getRank(col-1,cols,row-1);
#endif
        } else {
            mpi_ranks[0] = -55555;
#ifdef _OPENMP
            omp_ranks[0] = -55555;
#endif
        }

        //up right
#ifdef _OPENMP
        if(loc_col+1<loc_cols && loc_row-1>=0) {
            mpi_ranks[2] = mpi_rank;
            omp_ranks[2] = getRank(loc_col+1,loc_cols,loc_row-1);
        } else if((loc_col+1>=loc_cols || loc_row-1<0) && (col+1<cols && row-1>=0)) {
            unsigned int absRank = getRank(col+1,cols,row-1);
            mpi_ranks[2] = getMPIRank(absRank);
            omp_ranks[2] = getOMPRank(absRank);
#else
        if(col+1<cols && row-1>=0) {
            mpi_ranks[2] = getRank(col+1,cols,row-1);
#endif
        } else {
            mpi_ranks[2] = -55555;
#ifdef _OPENMP
            omp_ranks[2] = -55555;
#endif
        }

        //down left
#ifdef _OPENMP
        if(loc_col-1>=0 && loc_row+1<loc_rows) {
            mpi_ranks[6] = mpi_rank;
            omp_ranks[6] = getRank(loc_col-1,loc_cols,loc_row+1);
        } else if((loc_col-1<0 || loc_row+1>=loc_rows) && (col-1>=0 && row+1<rows)) {
            unsigned int absRank = getRank(col-1,cols,row+1);
            mpi_ranks[6] = getMPIRank(absRank);
            omp_ranks[6] = getOMPRank(absRank);
#else
        if(col-1>=0 && row+1<rows) {
            mpi_ranks[6] = getRank(col-1,cols,row+1);
#endif
        } else {
            mpi_ranks[6] = -55555;
#ifdef _OPENMP
            omp_ranks[6] = -55555;
#endif
        }

        //down right
#ifdef _OPENMP
        if(loc_col+1<loc_cols && loc_row+1<loc_rows) {
            mpi_ranks[4] = mpi_rank;
            omp_ranks[4] = getRank(loc_col+1,loc_cols,loc_row+1);
        } else if((loc_col+1>=loc_cols || loc_row+1>=loc_rows) && (col+1<cols && row+1<rows)) {
            unsigned int absRank = getRank(col+1,cols,row+1);
            mpi_ranks[4] = getMPIRank(absRank);
            omp_ranks[4] = getOMPRank(absRank);
#else
        if(col+1<cols && row+1<rows) {
            mpi_ranks[4] = getRank(col+1,cols,row+1);
#endif
        } else {
            mpi_ranks[4] = -55555;
#ifdef _OPENMP
            omp_ranks[4] = -55555;
#endif
        }
    }
    //-------------------------------------------------------------------------------

    //! get the border for the refCol and refRow neighbour
    /*!
      \param refCol is the reference column number to the neighbour whiwh ask the border
      \param refRow is the reference row number to the neighbour whiwh ask the border
      \param border is the border size to take into account
      \return list of values
    */
    //-------------------------------------------------------------------------------
    T * getBorderToSend(int refCol, int refRow)
    //-------------------------------------------------------------------------------
    {
        T * result=NULL;

        unsigned int sizeBB = border*border;
        unsigned int sizeWB = loc_head.width*border;
        unsigned int sizeHB = loc_head.height*border;
        if(refCol==col-1 && refRow==row-1) {
            result = new T[sizeBB];
            for(int i=0; i<border; i++) {
                for(int j=0; j<border; j++) {
                    unsigned int r1 = i*border+j;
                    //===========matrix with border
                    //unsigned int r2 = i*loc_head.width+j;
                    unsigned int r2 = (border_head.width*border + border) + i*border_head.width + j;
                    result[r1] = data[r2];
                }
            }
        } else if(refCol==col-1 && refRow==row) {
            result = new T[sizeHB];
            for(unsigned int i=0; i<loc_head.height; i++) {
                for(int j=0; j<border; j++) {
                    unsigned int r1 = i*border+j;
                    //===========matrix with border
                    //unsigned int r2 = (i*loc_head.width)+j;
                    unsigned int r2 = (border*border_head.width + border) + i*border_head.width + j;
                    result[r1] = data[r2];
                }
            }
        } else if(refCol==col-1 && refRow==row+1) {
            result = new T[sizeBB];
            for(int i=0; i<border; i++) {
                for(int j=0; j<border; j++) {
                    unsigned int r1 = i*border+j;
                    //===========matrix with border
                    //unsigned int r2 = (loc_head.height*loc_head.width)-((border-i)*loc_head.width)+j;
                    unsigned int r2 = (border_head.height*border_head.width - (border+border)*border_head.width + border) + i*border_head.width + j;
                    result[r1] = data[r2];
                }
            }
        } else if(refCol==col+1 && refRow==row-1) {
            result = new T[sizeBB];
            for(int i=0; i<border; i++) {
                for(int j=0; j<border; j++) {
                    unsigned int r1 = i*border+j;
                    //===========matrix with border
                    //unsigned int r2 = (((i+1)*loc_head.width)-1)-(border-1-j);
                    unsigned int r2 = ((border+1)*border_head.width - border*2) + i*border_head.width +j;
                    result[r1] = data[r2];
                }
            }
        } else if(refCol==col+1 && refRow==row) {
            result = new T[sizeHB];
            for(unsigned int i=0; i<loc_head.height; i++) {
                for(int j=0; j<border; j++) {
                    unsigned int r1 = i*border+j;
                    //===========matrix with border
                    //unsigned int r2 = (((i+1)*loc_head.width)-1)-(border-1-j);
                    unsigned int r2 = ((border+1)*border_head.width - border*2) + i*border_head.width +j;
                    result[r1] = data[r2];
                }
            }
        } else if(refCol==col+1 && refRow==row+1) {
            result = new T[sizeBB];
            for(int i=0; i<border; i++) {
                for(int j=0; j<border; j++) {
                    unsigned int r1 = i*border+j;
                    //===========matrix with border
                    //unsigned int r2 = (loc_head.height*loc_head.width)-((border-i-1)*loc_head.width)-(border-j);
                    unsigned int r2 = (border_head.height*border_head.width - (border+border-1)*border_head.width - border*2) + i*border_head.width +j;
                    result[r1] = data[r2];
                }
            }
        } else if(refCol==col && refRow==row-1) {
            result = new T[sizeWB];
            for(int i=0; i<border; i++) {
                for(unsigned int j=0; j<loc_head.width; j++) {
                    unsigned int r1 = i*loc_head.width+j;
                    //===========matrix with border
                    //unsigned int r2 = i*loc_head.width+j;
                    unsigned int r2 = (border*border_head.width + border) + i*border_head.width+j;
                    result[r1] = data[r2];
                }
            }
        } else if(refCol==col && refRow==row+1) {
            result = new T[sizeWB];
            for(int i=0; i<border; i++) {
                for(unsigned int j=0; j<loc_head.width; j++) {
                    unsigned int r1 = i*loc_head.width+j;
                    //===========matrix with border
                    //unsigned int r2 = (loc_head.width*loc_head.height)-((border-i)*loc_head.width)+j;
                    unsigned int r2 = (border_head.height*border_head.width - (border+border)*border_head.width + border) + i*border_head.width + j;
                    result[r1] = data[r2];
                }
            }
        }

        return result;
    }
    //-------------------------------------------------------------------------------


    //! used by get borders to make exchanges
    /*!
      \param toGet is the data to get from mpi process and omp thread
      \param toSend is the data to send to mpi process and omp thread
      \param size is the size of data to send and recv
      \param mpi is the mpi rank to exchange with
      \param omp is the omp rank to exchange with
      \param s boolean : lock or unlock the mpi communications
    */
    void BorderExchanges(T * toGet, T* toSend,unsigned int size,int mpi, int omp, int s)
    //-------------------------------------------------------------------------------
    {
        MPI_Status status;
        MPI_Request request;

#ifdef DEBUG
        std::stringstream st;
        st<<"Values to give to rank "<<mpi<<"\n";
        for(int j=0; j<size; j++) {
            st<<toSend[j]<<" ";
        }
        st<<"\n";
        Mpiomp::print(st.str());
#endif

#ifdef _OPENMP
        std::stringstream send_tagstream;
        send_tagstream<<omp<<omp_rank;
        int send_tag = atoi(send_tagstream.str().c_str());
        std::stringstream recv_tagstream;
        recv_tagstream<<omp_rank<<omp;
        int recv_tag = atoi(recv_tagstream.str().c_str());

        if(s==0) {
            MPI_Send((void *)toSend,size,MPI_DOUBLE,mpi,send_tag,MPI_COMM_WORLD);
            MPI_Recv(toGet,size,MPI_DOUBLE,mpi,recv_tag,MPI_COMM_WORLD,&status);
        } else {
            MPI_Recv(toGet,size,MPI_DOUBLE,mpi,recv_tag,MPI_COMM_WORLD,&status);
            MPI_Send((void *)toSend,size,MPI_DOUBLE,mpi,send_tag,MPI_COMM_WORLD);
        }
#else
        if(s==0) {
            MPI_Send((void *)toSend,size,MPI_DOUBLE,mpi,0,MPI_COMM_WORLD);
            MPI_Recv(toGet,size,MPI_DOUBLE,mpi,MPI_ANY_TAG,MPI_COMM_WORLD,&status);
        } else {
            MPI_Recv(toGet,size,MPI_DOUBLE,mpi,MPI_ANY_TAG,MPI_COMM_WORLD,&status);
            MPI_Send((void *)toSend,size,MPI_DOUBLE,mpi,0,MPI_COMM_WORLD);
        }
#endif
#ifdef DEBUG
        std::stringstream st2;
        st2<<"values given by rank "<<mpi<<" and omp "<<omp<<"\n";
        for(int j=0; j<size; j++) {
            st2<<toGet[j]<<" ";
        }
        st2<<"\n";
        Mpiomp::print(st2.str());
#endif
    }
    //-------------------------------------------------------------------------------

    //! used by getBorders to fill results of exchanges with the neighbours processors (by MPI exchanges)
    //! It has to fill data
    /*!
      \param position is the border type of the data (0=upl, 1=up, 2=upr, 3=right, 4=downr, 5=down, 6=downl, 7=left)
      \param values is the table containing the data of the border
    */
    void FillBorders(int position, T * values)
    //-------------------------------------------------------------------------------
    {
#ifdef DEBUG
        std::stringstream st;
        st<<"In FillBorders\n";
        Mpiomp::print(st.str());
#endif

        switch (position) {
        case 0:
            for(int i=0; i<border; i++) {
                for(int j=0; j<border; j++)
                    data[i*border_head.width+j] = values[i*border+j];
            }
            break;
        case 1:
            for(int i=0; i<border; i++) {
                for(int j=0; j<loc_head.width; j++)
                    data[(border) + i*border_head.width+j] = values[i*loc_head.width+j];
            }
            break;
        case 2:
            for(int i=0; i<border; i++) {
                for(int j=0; j<border; j++)
                    data[(border_head.width-border) + i*border_head.width+j] = values[i*border+j];
            }
            break;
        case 3:
            for(int i=0; i<loc_head.height; i++) {
                for(int j=0; j<border; j++)
                    data[(border_head.width*border + border_head.width - border) + i*border_head.width + j] = values[i*border+j];
            }
            break;
        case 4:
            for(int i=0; i<border; i++) {
                for(int j=0; j<border; j++)
                    data[(border_head.width*border_head.height - (border-1)*border_head.width - border) + i*border_head.width + j] = values[i*border+j];
            }
            break;
        case 5:
            for(int i=0; i<border; i++) {
                for(int j=0; j<loc_head.width; j++)
                    data[(border_head.width*border_head.height - border_head.width*border + border) + i*border_head.width +j] = values[i*loc_head.width+j];
            }
            break;
        case 6:
            for(int i=0; i<border; i++) {
                for(int j=0; j<border; j++)
                    data[(border_head.width*border_head.height - border_head.width*border) + i*border_head.width +j] = values[i*border+j];
            }
            break;
        case 7:
            for(int i=0; i<loc_head.height; i++) {
                for(int j=0; j<border; j++)
                    data[(border_head.width*border) + i*border_head.width + j] = values[i*border+j];
            }
        }
    }
    //-------------------------------------------------------------------------------

    //===================================================================================//

    //=======================GET 8 NEIGHBORS METHODS=======================//
    //get the 8 neighbors for the left Up element of the matrix
    //-------------------------------------------------------------------------------
    /* void leftUpNeighbors(T * result, unsigned int rank)
      //-------------------------------------------------------------------------------
      {
        unsigned int itR = rank;

        if(NULL!=uplValues)
    {
      result[0] = uplValues[(border*border)-1];
    }

    #ifdef _OPENMP
        else if(list!=NULL && loc_col-1>=0 && loc_row-1>=0)
    {
      int omp = (loc_row-1)*loc_cols+(loc_col-1);
      Matrix<T> * proc = (Matrix<T> *)list[omp];
      unsigned int r0 = proc->getLocalSize()-1;
      result[0] = proc->getValue(r0);
    }
    #endif

        else
    result[0] = -9999;
        if(NULL!=upValues)
    {
      unsigned int r1 = (border-1)*loc_head.width;
      unsigned int r2 = (border-1)*loc_head.width+1;
      result[1] = upValues[r1];
      result[2] = upValues[r2];
    }

    #ifdef _OPENMP
        else if(list!=NULL && loc_row-1>=0)
    {
      int omp = (loc_row-1)*loc_cols+loc_col;
      Matrix<T> * proc = (Matrix<T> *)list[omp];
      unsigned int r1 = proc->getLocalSize()-proc->getLocalWidth();
      unsigned int r2 = proc->getLocalSize()-proc->getLocalWidth()+1;
      result[1] = proc->getValue(r1);
      result[2] = proc->getValue(r2);
    }
    #endif
        else
    {
      result[1] = -9999;
      result[2] = -9999;
    }
        unsigned int r3 = itR+1;
        unsigned int r4 = itR+loc_head.width+1;
        unsigned int r5 = itR+loc_head.width;
        result[3] = data[r3];
        result[5] = data[r5];
        result[4] = data[r4];
        //=============no matrix.h
        //result[3] = subMatrix->getValue(r3);
        //result[5] = subMatrix->getValue(r5);
        //result[4] = subMatrix->getValue(r4);

        if(NULL!=leftValues)
    {
      result[7] = leftValues[border-1];
      result[6] = leftValues[(border*2)-1];
    }

    #ifdef _OPENMP
        else if(list!=NULL && loc_col-1>=0)
    {
      int omp = loc_row*loc_cols+(loc_col-1);
      Matrix<T> * proc = (Matrix<T> *)list[omp];
      unsigned int r6 = proc->getLocalWidth()*2-1;
      unsigned int r7 = proc->getLocalWidth()-1;
      result[7] = proc->getValue(r7);
      result[6] = proc->getValue(r6);
    }
    #endif
        else
    {
      result[6] = -9999;
      result[7] = -9999;
    }
    }*/
    //-------------------------------------------------------------------------------

    //get the 8 neighbors for the left down element of the matrix
    //-------------------------------------------------------------------------------
    /*void leftDownNeighbors(T * result, unsigned int rank)
      //-------------------------------------------------------------------------------
      {
        unsigned int itR= rank;

        if(NULL!=leftValues)
    {
      unsigned int r7 = (loc_head.height*border)-1;
      unsigned int r0 = ((loc_head.height-1)*border)-1;
      result[0] = leftValues[r0];
      result[7] = leftValues[r7];
      }
    #ifdef _OPENMP
        else if(list!=NULL && loc_col-1>=0)
    {
      int omp = loc_row*loc_cols+(loc_col-1);
      Matrix<T> * proc = (Matrix<T> *)list[omp];
      unsigned int r7 = proc->getLocalSize()-1;
      unsigned int r0 = proc->getLocalSize()-1-proc->getLocalWidth();
      result[0] = proc->getValue(r0);
      result[7] = proc->getValue(r7);
    }
    #endif
        else
    {
      result[0] = -9999;
      result[7] = -9999;
    }

        unsigned int r1 = itR-loc_head.width;
        unsigned int r2 = itR-loc_head.width+1;
        unsigned int r3 = itR+1;
        result[1] = data[r1];
        result[2] = data[r2];
        result[3] = data[r3];
        //=============no matrix.h
        //result[1] = subMatrix->getValue(r1);
        //result[2] = subMatrix->getValue(r2);
        //result[3] = subMatrix->getValue(r3);

        if(NULL!=downlValues)
        result[6] = downlValues[border-1];

    #ifdef _OPENMP
        else if(list!=NULL && loc_row+1<loc_rows && loc_col-1>=0)
    {
      int omp = (loc_row+1)*loc_cols+(loc_col-1);
      Matrix<T> * proc = (Matrix<T> *)list[omp];
      unsigned int r6 = proc->getLocalWidth()-1;
      result[6] = proc->getValue(r6);
    }
    #endif
        else
    result[6] = -9999;

        if(NULL!=downValues)
    {
      result[5] = downValues[0];
      result[4] = downValues[1];
      }
    #ifdef _OPENMP
        else if(list!=NULL && loc_row+1<loc_rows)
    {
      int omp = (loc_row+1)*loc_cols+loc_col;
      Matrix<T> * proc = (Matrix<T> *)list[omp];
      result[5] = proc->getValue(0);
      result[4] = proc->getValue(1);
    }
    #endif
        else
    {
      result[4] = -9999;
      result[5] = -9999;
    }

    }*/
    //-------------------------------------------------------------------------------

    //get the 8 neighbors for the right down element of the matrix
    //-------------------------------------------------------------------------------
    /*void rightDownNeighbors(T * result, unsigned int rank)
    //-------------------------------------------------------------------------------
    {
      unsigned int itR= rank;

      unsigned int r0 = itR-loc_head.width-1;
      unsigned int r1 = itR-loc_head.width;
      unsigned int r7 = itR-1;
      result[0] = data[r0];
      result[1] = data[r1];
      result[7] = data[r7];
      //=============no matrix.h
      //result[0] = subMatrix->getValue(r0);
      //result[1] = subMatrix->getValue(r1);
      //result[7] = subMatrix->getValue(r7);
      if(NULL!=rightValues)
    {
      unsigned int r2 = loc_head.height*border-2*border;
      unsigned int r3 = loc_head.height*border-border;
      result[2] = rightValues[r2];
      result[3] = rightValues[r3];
      }
    #ifdef _OPENMP
      else if(list!=NULL && loc_col+1<loc_cols)
    {
      int omp = loc_row*loc_cols+(loc_col+1);
      Matrix<T> * proc = (Matrix<T> *)list[omp];
      unsigned int r2 = proc->getLocalSize()-(proc->getLocalWidth()*2);
      unsigned int r3 = proc->getLocalSize()-proc->getLocalWidth();
      result[2] = proc->getValue(r2);
      result[3] = proc->getValue(r3);
    }
    #endif
      else
    {
      result[2] = -9999;
      result[3] = -9999;
    }

      if(NULL!=downValues)
    {
      unsigned int r5 = loc_head.width-1;
      unsigned int r6 = loc_head.width-2;
      result[6] = downValues[r6];
      result[5] = downValues[r5];
      }
    #ifdef _OPENMP
      else if(list!=NULL && loc_row+1<loc_rows)
    {
      int omp = (loc_row+1)*loc_cols+loc_col;
      Matrix<T> * proc = (Matrix<T> *)list[omp];
      unsigned int r5 = proc->getLocalWidth()-1;
      unsigned int r6 = proc->getLocalWidth()-2;
      result[6] = proc->getValue(r6);
      result[5] = proc->getValue(r5);
    }
    #endif
      else
    {
      result[5] = -9999;
      result[6] = -9999;
    }

      if(NULL!=downrValues)
    result[4] = downrValues[0];

    #ifdef _OPENMP
      else if(list!=NULL && loc_row+1<loc_rows && loc_col+1<loc_cols)
    {
      int omp = (loc_row+1)*loc_cols+(loc_col+1);
      Matrix<T> * proc = (Matrix<T> *)list[omp];
      result[4] = proc->getValue(0);
    }
    #endif
      else
    result[4] = -9999;

    }*/
    //-------------------------------------------------------------------------------

    //get the 8 neighbors for the right up element of the matrix
    //-------------------------------------------------------------------------------
    /*void rightUpNeighbors(T * result, unsigned int rank)
    //-------------------------------------------------------------------------------
    {
      unsigned int itR= rank;

      if(NULL!=upValues)
    {
      unsigned int r0 = loc_head.width*(border-1)+loc_head.width-2;
      unsigned int r1 = loc_head.width*(border-1)+loc_head.width-1;
      result[0] = upValues[r0];
      result[1] = upValues[r1];
      }
    #ifdef _OPENMP
      else if(list!=NULL && loc_row-1>=0)
    {
      int omp = (loc_row-1)*loc_cols+loc_col;
      Matrix<T> * proc = (Matrix<T> *)list[omp];
      unsigned int r0 = proc->getLocalSize()-2-offset_decomp;
      unsigned int r1 = proc->getLocalSize()-1-offset_decomp;
      result[0] = proc->getValue(r0);
      result[1] = proc->getValue(r1);
    }
    #endif
      else
    {
      result[0] = -9999;
      result[1] = -9999;
    }

      if(NULL!=uprValues)
      result[2] = uprValues[border*(border-1)];

    #ifdef _OPENMP
      else if(list!=NULL && loc_row-1>=0 && loc_col+1<loc_cols)
    {
      int omp = (loc_row-1)*loc_cols+(loc_col+1);
      Matrix<T> * proc = (Matrix<T> *)list[omp];
      unsigned int r2 = proc->getLocalSize()-proc->getLocalWidth();
      result[2] = proc->getValue(r2);
    }
    #endif
      else
    result[2] = -9999;

      unsigned int r5 = itR+loc_head.width;
      unsigned int r6 = itR+loc_head.width-1;
      unsigned int r7 = itR-1;
      result[6] = data[r6];
      result[5] = data[r5];
      result[7] = data[r7];
      //=============no matrix.h
      //result[6] = subMatrix->getValue(r6);
      //result[5] = subMatrix->getValue(r5);
      //result[7] = subMatrix->getValue(r7);

      if(NULL!=rightValues)
    {
      result[3] = rightValues[0];
      result[4] = rightValues[border];
      }

    #ifdef _OPENMP
      else if(list!=NULL && loc_col+1<loc_cols)
    {
      int omp = loc_row*loc_cols+(loc_col+1);
      Matrix<T> * proc = (Matrix<T> *)list[omp];
      result[3] = proc->getValue(0);
      unsigned int r4 = proc->getLocalWidth();
      result[4] = proc->getValue(r4);
    }
    #endif
      else
    {
      result[3] = -9999;
      result[4] = -9999;
    }

    }*/
    //-------------------------------------------------------------------------------

    //get the 8 neighbors an element on the left border of the matrix
    //-------------------------------------------------------------------------------
    /* void leftNeighbors(T * result, unsigned int rank)
      //-------------------------------------------------------------------------------
      {
        unsigned int itR= rank;
        if(NULL!=leftValues)
    {
      unsigned int height = floor(itR/loc_head.width);
      unsigned int r0 = height*border-1;
      unsigned int r7 = height*border+(border-1);
      unsigned int r6 = (height+1)*border+(border-1);
      result[0] = leftValues[r0];
      result[7] = leftValues[r7];
      result[6] = leftValues[r6];
      }

    #ifdef _OPENMP
        else if(list!=NULL && loc_col-1>=0)
    {
      int omp = loc_row*loc_cols+(loc_col-1);
      Matrix<T> * proc = (Matrix<T> *)list[omp];
      unsigned int height = floor(itR/proc->getLocalWidth());
      unsigned int r0 = height*proc->getLocalWidth()-1;
      unsigned int r7 = height*proc->getLocalWidth()+(proc->getLocalWidth()-1);
      unsigned int r6 = (height+1)*proc->getLocalWidth()+(proc->getLocalWidth()-1);
      result[0] = proc->getValue(r0);
      result[7] = proc->getValue(r7);
      result[6] = proc->getValue(r6);
    }
    #endif
        else
    {
      result[0] = -9999;
      result[6] = -9999;
      result[7] = -9999;
    }

        unsigned int r1 = itR-loc_head.width;
        unsigned int r2 = itR-loc_head.width+1;
        unsigned int r3 = itR+1;
        unsigned int r4 = itR+loc_head.width+1;
        unsigned int r5 = itR+loc_head.width;
        result[1] = data[r1];
        result[2] = data[r2];
        result[3] = data[r3];
        result[5] = data[r5];
        result[4] = data[r4];
        //=============no matrix.h
        //result[1] = subMatrix->getValue(r1);
        //result[2] = subMatrix->getValue(r2);
        //result[3] = subMatrix->getValue(r3);
        //result[5] = subMatrix->getValue(r5);
        //result[4] = subMatrix->getValue(r4);

        //cache optimization for next neighbors
        //precedent8N[0] = result[0];
    //precedent8N[1] = result[1];
    //precedent8N[2] = result[2];
    //precedent8N[3] = result[3];
    //precedent8N[4] = result[4];
    //precedent8N[5] = result[5];
    //precedent8N[6] = result[6];
    //precedent8N[7] = result[7];
    //precedent8NRank=rank;
    }*/
    //-------------------------------------------------------------------------------

    //get the 8 neighbors an element on the right border of the matrix
    //-------------------------------------------------------------------------------
    /*void rightNeighbors(T * result, unsigned int rank)
    //-------------------------------------------------------------------------------
    {
      unsigned int itR= rank;
      //cache optimization : We can use precedent8N but not fill it
      //if(precedent8NRank+1==rank)
    //{
    //result[0]=precedent8N[1];
    //result[1]=precedent8N[2];
    //result[6]=precedent8N[5];
    //result[5]=precedent8N[4];
    //result[7]=subMatrix->getValue(precedent8NRank);
    //}
    //else
    //{
      unsigned int r0 = itR-loc_head.width-1;
      unsigned int r1 = itR-loc_head.width;
      unsigned int r6 = itR+loc_head.width-1;
      unsigned int r5 = itR+loc_head.width;
      unsigned int r7 = itR-1;
      result[0] = data[r0];
      result[1] = data[r1];
      result[6] = data[r6];
      result[5] = data[r5];
      result[7] = data[r7];
      //=============no matrix.h
      //result[0] = subMatrix->getValue(r0);
      //result[1] = subMatrix->getValue(r1);
      //result[6] = subMatrix->getValue(r6);
      //result[5] = subMatrix->getValue(r5);
      //result[7] = subMatrix->getValue(r7);
      //	}
      if(NULL!=rightValues)
    {
      unsigned int height = floor(itR/loc_head.width);
      unsigned int r2 = (height-1)*border;
      unsigned int r3 = height*border;
      unsigned int r4 = (height+1)*border;
      result[3] = rightValues[r3];
      result[2] = rightValues[r2];
      result[4] = rightValues[r4];
      }
    #ifdef _OPENMP
      else if(list!=NULL && loc_col+1<loc_cols)
    {
      int omp = loc_row*loc_cols+(loc_col+1);
      Matrix<T> * proc = (Matrix<T> *)list[omp];
      unsigned int height = floor(itR/proc->getLocalWidth());
      unsigned int r2 = (height-1)*proc->getLocalWidth();
      unsigned int r3 = height*proc->getLocalWidth();
      unsigned int r4 = (height+1)*proc->getLocalWidth();
      result[3] = proc->getValue(r3);
      result[2] = proc->getValue(r2);
      result[4] = proc->getValue(r4);
    }
    #endif
      else
    {
      result[2] = -9999;
      result[3] = -9999;
      result[4] = -9999;
    }
    }*/
    //-------------------------------------------------------------------------------


    //get the 8 neighbors an element on the up border of the matrix
    //-------------------------------------------------------------------------------
    /* void upNeighbors(T * result, unsigned int rank)
      //-------------------------------------------------------------------------------
      {
        unsigned int itR= rank;

        if(NULL!=upValues)
    {
      unsigned int r0 = loc_head.width*(border-1)+itR-1;
      unsigned int r1 = loc_head.width*(border-1)+itR;
      unsigned int r2 = loc_head.width*(border-1)+itR+1;
      if(precedent8NRank+1==rank)
        {
          result[0] = precedent8N[1];
          result[1]=precedent8N[2];
        }
      else
        {
          result[0] = upValues[r0];
          result[1] = upValues[r1];
        }
      result[2] = upValues[r2];
      }

    #ifdef _OPENMP
        else if(list!=NULL && loc_row-1>=0)
    {
      int omp = (loc_row-1)*loc_cols+loc_col;
      Matrix<T> * proc = (Matrix<T> *)list[omp];

      if(precedent8NRank+1==rank)
        {
          result[0] = precedent8N[1];;
          result[1]=precedent8N[2];
        }
      else
        {
          unsigned int r0 = proc->getLocalSize()-proc->getLocalWidth()+itR-1;
          unsigned int r1 = proc->getLocalSize()-proc->getLocalWidth()+itR;
          result[0] = proc->getValue(r0);
          result[1] = proc->getValue(r1);
        }

      unsigned int r2 = proc->getLocalSize()-proc->getLocalWidth()+itR+1;
      result[2] = proc->getValue(r2);
    }
    #endif
        else
    {
      result[0] = -9999;
      result[1] = -9999;
      result[2] = -9999;
    }

        if(precedent8NRank+1==rank)
    {
      result[6]=precedent8N[5];
      result[5]=precedent8N[4];
      result[7] = data[precedent8NRank];
      //=============no matrix.h
      //result[7]=subMatrix->getValue(precedent8NRank);
    }
        else
    {
      unsigned int r5 = itR+loc_head.width;
      unsigned int r6 = itR+loc_head.width-1;
      unsigned int r7 = itR-1;
      result[6] = data[r6];
      result[5] = data[r5];
      result[7] = data[r7];
      //=============no matrix.h
      //result[6] = subMatrix->getValue(r6);
      //result[5] = subMatrix->getValue(r5);
      //result[7] = subMatrix->getValue(r7);
    }

        unsigned int r3 = itR+1;
        unsigned int r4 = itR+loc_head.width+1;
        result[3] = data[r3];
        result[4] = data[r4];
        //=============no matrix.h
        //result[3] = subMatrix->getValue(r3);
    //result[4] = subMatrix->getValue(r4);
        //cache optimization for next neighbors
        precedent8N[0] = result[0];
        precedent8N[1] = result[1];
        precedent8N[2] = result[2];
        precedent8N[3] = result[3];
        precedent8N[4] = result[4];
        precedent8N[5] = result[5];
        precedent8N[6] = result[6];
        precedent8N[7] = result[7];
        precedent8NRank=rank;

        }*/
    //-------------------------------------------------------------------------------

    //get the 8 neighbors an element on the up border of the matrix
    //-------------------------------------------------------------------------------
    /*void downNeighbors(T * result, unsigned int rank)
    //-------------------------------------------------------------------------------
    {
      unsigned int itR= rank;
      //cache optimization
      if(precedent8NRank+1==rank)
    {
      result[0] = precedent8N[1];
      result[1]=precedent8N[2];
    }
      else
    {
      unsigned int r0 = itR-loc_head.width-1;
      unsigned int r1 = itR-loc_head.width;
      result[0] = data[r0];
      result[1] = data[r1];
      //=============no matrix.h
      //result[0] = subMatrix->getValue(r0);
      //result[1] = subMatrix->getValue(r1);
    }

      unsigned int r2 = itR-loc_head.width+1;
      unsigned int r3 = itR+1;
      result[2] = data[r2];
      //=============no matrix.h
      //result[2] = subMatrix->getValue(r2);

      //cache optimization
      if(precedent8NRank+1==rank)
    result[7] = data[precedent8NRank];
    //=============no matrix.h
    //result[7]=subMatrix->getValue(precedent8NRank);
      else
    {
      unsigned int r7 = itR-1;
      result[7] = data[r7];
      //=============no matrix.h
      //result[7] = subMatrix->getValue(r7);
    }

      result[3] = data[r3];
      //=============no matrix.h
      //result[3] = subMatrix->getValue(r3);

      if(NULL!=downValues)
    {
      unsigned int w = itR-(loc_head.width*(loc_head.height-1));
      //cache optimization
      if(precedent8NRank+1==rank)
        {
          result[6]=precedent8N[5];
          result[5]=precedent8N[4];
        }
      else
        {
          result[6] = downValues[w-1];
          result[5] = downValues[w];
        }

      result[4] = downValues[w+1];
      }

    #ifdef _OPENMP
      else if(list!=NULL && loc_row+1<loc_rows)
    {
      int omp = (loc_row+1)*loc_cols+loc_col;
      Matrix<T> * proc = (Matrix<T> *)list[omp];
      unsigned int w = itR-(proc->getLocalWidth()*(proc->getLocalHeight()-1));

      //cache optimization
      if(precedent8NRank+1==rank)
        {
          result[6]=precedent8N[5];
          result[5]=precedent8N[4];
        }
      else
        {
          result[6] = proc->getValue(w-1);
          result[5] = proc->getValue(w);
        }

      result[4] = proc->getValue(w+1);
    }
    #endif
      else
    {
      result[4] = -9999;
      result[5] = -9999;
      result[6] = -9999;
    }


      //cache optimization for next neighbors
      precedent8N[0] = result[0];
      precedent8N[1] = result[1];
      precedent8N[2] = result[2];
      precedent8N[3] = result[3];
      precedent8N[4] = result[4];
      precedent8N[5] = result[5];
      precedent8N[6] = result[6];
      precedent8N[7] = result[7];
      precedent8NRank=rank;
      }*/
    //-------------------------------------------------------------------------------

    //! get the 8 Neighbors in case the iterator is in the middle
    /*!
      \param result is the table with the 8 neighbors values
      \param rank is the rank of the element to analyse
    */
    //-------------------------------------------------------------------------------
    /* void middleNeighbors(T * result, unsigned int rank)
    //-------------------------------------------------------------------------------
    {
      subMatrix->getNeighborsValues(rank,result);
      }*/
    //-------------------------------------------------------------------------------

    //=======================================================================//
    //! iterator modificator
    /*!
      This constructor will allocate the good space, the matrix will not be filled but will be ready to be filled.
      \param value is the new value for the returned iterator
      \return an iterator with the good rank and the good value
    */
    iterator assignIt_Rank(int rank) {
        iterator it;
        //=============no matrix.h
        //it = subMatrix->getValuePtr(rank);
        //it = &data[rank];
        it.rank=rank;
        it.border=border;
        it.width=border_head.width;
        return it;
    }


public:

    //! constructor of the distributed matrix
    /*!
      This constructor will simply allocate the good space, the matrix will not be filled but will be ready to be filled.
      \param h is the header you want to use to build your matrix
      \param local say if the matrix created will or won't be local
    */
    //-------------------------------------------------------------------------------
    DMatrix(HEADER h, int b=0, int value=-1, bool loc=false)
    //-------------------------------------------------------------------------------
    {
        init(loc,b);
        head=h;
        createSubMatrice((const char *)( ""), false, value);
        /*
        #ifdef _OPENMP
        if(loc==false)
          //=============no matrix.h
          //NE MARCHE PLUS
          shared_mem =  Mpiomp::AddListMatrix((MatrixBase *)subMatrix,omp_rank);
          #endif*/
    }

    //! constructor of the distributed matrix
    /*!
      The constructor evaluate how to divide the matrix in submatrice and do the work.
      \param binFile is the path of the binary file you want to work on
    */
    //-------------------------------------------------------------------------------
    DMatrix(const char * binFile, int b=0)
    //-------------------------------------------------------------------------------
    {
        init(false,b);
        readHeader(binFile);
        //read(binFile);
        /*
          #ifdef _OPENMP
          #pragma omp master
          {
          #endif
          read(binFile);
          #ifdef _OPENMP
          }
          #pragma omp barrier
          #endif*/

        createSubMatrice(binFile, true);
        /*
        #ifdef _OPENMP
        //=============no matrix.h
        //NE MARCHE PLUS
        shared_mem = Mpiomp::AddListMatrix((MatrixBase *)subMatrix,omp_rank);
        #endif*/
    }
    //-------------------------------------------------------------------------------

    //! destructor of the distributed matrix
    /*!  */
    //-------------------------------------------------------------------------------
    ~DMatrix()
    //-------------------------------------------------------------------------------
    {
        if(local)
            //=============no matrix.h
            //delete subMatrix;
            delete [] data;
        else {
#ifdef _OPENMP
#pragma omp barrier
#endif
            //=============no matrix.h
            //delete subMatrix;
            delete [] data;
        }

        /*if(NULL!=upValues)
          delete [] upValues;
        if(NULL!=downValues)
          delete [] downValues;
        if(NULL!=leftValues)
          delete [] leftValues;
        if(NULL!=rightValues)
          delete [] rightValues;
        if(NULL!=uplValues)
          delete [] uplValues;;
        if(NULL!=uprValues)
          delete [] uprValues;
        if(NULL!=downlValues)
          delete [] downlValues;
        if(NULL!=downrValues)
        delete [] downrValues;*/
        if(NULL!=precedent8N)
            delete [] precedent8N;
        /*if(NULL!=result)
          delete [] result;*/
        /*#pragma omp master
          {
          delete list; list=NULL;
          }*/
    }
    //-------------------------------------------------------------------------------

    //=======================USER METHODS=======================//

    //! begin iterator of the matrix
    /*!
      \return returns the iterator on the beginning of the matrix
    */
    iterator begin()
    //-------------------------------------------------------------------------------
    {
        //=============no matrix.h
        //if(NULL!=subMatrix)
        if(NULL!=data)
            return assignIt_Rank(border*border_head.width+border);
        //===========matrix with border
        //return assignIt_Rank(0);
    }
    //-------------------------------------------------------------------------------

    //! end iterator of the matrix
    /*!
      \return returns the iterator on the end of the matrix
    */
    iterator end()
    //-------------------------------------------------------------------------------
    {
        //=============no matrix.h
        //if(NULL!=subMatrix)
        //return assignIt_Rank(subMatrix->getLocalSize()-1);

        //TODO + OFFSET
        if(NULL!=data)
            return assignIt_Rank((border_head.width*border_head.height)-1-border*border_head.width-border);
        //===========matrix with border
        //return assignIt_Rank((loc_head.width*loc_head.height)-1);
    }
    //-------------------------------------------------------------------------------

    //! get the iterator on the matrix at position (x,y)
    /*!
      \param x is row index
      \pram y is column index
      \return returns the iterator at position (i,j)
    */
    iterator getIterator(int x,int y)
    //-------------------------------------------------------------------------------
    {
        if(NULL!=data) {
            unsigned int rank = border*border_head.width + border + y*border_head.width+x;
            //if(rank<border_head.width*border_head.height)
            return assignIt_Rank(rank);
        }
    }
    //-------------------------------------------------------------------------------

    //! get the indexes of the iterator on the matrix
    /*!
      \param it is the iterator position asked
      \param x is the iterator position x returned
      \param y is the iterator position y returned
    */
    void getIndexes(iterator it,int &x,int &y)
    //-------------------------------------------------------------------------------
    {
        unsigned int relative_rank = it.rank - border*border_head.width - (floor(it.rank/border_head.width)-border)*border*2 -border;
        y = floor(relative_rank/loc_head.width);
        x = relative_rank%loc_head.width;
    }
    //-------------------------------------------------------------------------------


    //! Set value of the current iterator
    /*!
      \param val is the value to affect
    */
    //-------------------------------------------------------------------------------
    void setValue(T val, iterator it)
    //-------------------------------------------------------------------------------
    {
        /*#ifdef _OPENMP
        	#pragma omp flush (list)
        	#endif*/

        //===========matrix with border
        //offset is manage in the iterator rank
        unsigned int itR = it.getRank();
        data[itR] = val;
        //=============no matrix.h
        //subMatrix->setValue(itR, val);
        /*#ifdef _OPENMP
        	#pragma omp flush (list)
        	#endif*/
    }
    //-------------------------------------------------------------------------------

    //! Set value at a global position in the matrix
    /*!
      \param val is the value to affect
      \param i is the row index value
      \param j is the col index value
    */
    //-------------------------------------------------------------------------------
    void setGlobalValue(T val, unsigned int i, unsigned int j)
    //-------------------------------------------------------------------------------
    {
        //it's me !
        if(col*loc_head.width<=j && (col+1)*loc_head.width>j && row*loc_head.height<=i && (row+1)*loc_head.height>i) {
            //remettre en coordonnées locales
            unsigned int j_loc = j-col*loc_head.width;
            unsigned int i_loc = i-row*loc_head.height;
            unsigned int rank = border*border_head.width +  i_loc*border_head.width + j_loc + border;
            data[rank] = val;
        }
    }
    //-------------------------------------------------------------------------------

    //! Set value for borders of the matrix
    /*!
      \param val is the value to set to the borders
    */
    //-------------------------------------------------------------------------------
    void setBorderValue(T val)
    //-------------------------------------------------------------------------------
    {
        //up
        for(int i=0; i<border*border_head.width; i++)
            data[i] = val;
        //left
        for(int i=(border-1); i<(border_head.height-border); i++) {
            data[i*border_head.width] = val;
            data[i*border_head.width+1] = val;
        }
        //right
        for(int i=border; i<(border_head.height-border+1); i++) {
            data[i*border_head.width-border] = val;
            data[i*border_head.width-border+1] = val;
        }
        //down
        for(int i=border_head.width*border_head.height-border*border_head.width; i<border_head.width*border_head.height; i++)
            data[i] = val;
    }
    //-------------------------------------------------------------------------------

    //! Set value for physical borders of the matrix
    /*!
      \param val is the value to set to the physical borders
    */
    //-------------------------------------------------------------------------------
    void setPhysicalValue(T val)
    //-------------------------------------------------------------------------------
    {
        //up
        if(row==0) {
            for(int i=0; i<border*border_head.width; i++)
                data[i] = val;
        }
        //left
        if(col==0) {
            for(int i=0; i<border_head.height; i++) {
                data[i*border_head.width] = val;
                data[i*border_head.width+1] = val;
            }
        }
        //right
        if(col==cols-1) {
            for(int i=0; i<border_head.height; i++) {
                data[(i+1)*border_head.width-border] = val;
                data[(i+1)*border_head.width-border+1] = val;
            }
        }
        //down
        if(row==rows-1) {
            for(int i=border_head.width*border_head.height-border*border_head.width; i<border_head.width*border_head.height; i++)
                data[i] = val;
        }
    }
    //-------------------------------------------------------------------------------

    //! Set value for right physical borders of the matrix
    /*!
      \param val is the value to set to the right physical borders
    */
    //-------------------------------------------------------------------------------
    void setPhysicalRightValue(T val, iterator it)
    //-------------------------------------------------------------------------------
    {
        if(col==cols-1) {
            for(int i=1; i<=border; i++) {
                data[it.rank+i]=val;
            }
        }
    }
    //-------------------------------------------------------------------------------

    //! Set value for left physical borders of the matrix
    /*!
      \param val is the value to set to the left physical borders
    */
    //-------------------------------------------------------------------------------
    void setPhysicalLeftValue(T val, iterator it)
    //-------------------------------------------------------------------------------
    {
        //left
        if(col==0) {
            for(int i=1; i<=border; i++) {
                data[it.rank-i]=val;
            }
        }
    }
    //-------------------------------------------------------------------------------

    //! Set value for up physical borders of the matrix
    /*!
      \param val is the value to set to the up physical borders
    */
    //-------------------------------------------------------------------------------
    void setPhysicalUpValue(T val, iterator it)
    //-------------------------------------------------------------------------------
    {
        if(row==0) {
            for(int i=1; i<=border; i++) {
                data[it.rank-i*border_head.width]=val;
            }
        }
    }
    //-------------------------------------------------------------------------------

    //! Set value for down physical borders of the matrix
    /*!
      \param val is the value to set to the down physical borders
    */
    //-------------------------------------------------------------------------------
    void setPhysicalDownValue(T val, iterator it)
    //-------------------------------------------------------------------------------
    {
        //down
        if(row==rows-1) {
            for(int i=1; i<=border; i++) {
                data[it.rank+i*border_head.width]=val;
            }
        }
    }
    //-------------------------------------------------------------------------------

    //! Get the value at iterator i of the matrix
    /*!
      \param it is the iterator to get the value
      \return the value
    */
    //-------------------------------------------------------------------------------
    T getValue(iterator it)
    //-------------------------------------------------------------------------------
    {
        //===========matrix with border
        //offset is manage in the iterator rank
        return data[it.getRank()];
        //=============no matrix.h
        //return subMatrix->getValue(i);
    }
    //-------------------------------------------------------------------------------

    //! Get the value at rank i of the matrix
    /*!
      \param i is the rank to get the value
      \return the value
    */
    //-------------------------------------------------------------------------------
    T getValue(unsigned int i)
    //-------------------------------------------------------------------------------
    {
        return data[i];
    }
    //-------------------------------------------------------------------------------

    //-------------------------------------------------------------------------------
    T * getValuePtr(iterator it)
    //-------------------------------------------------------------------------------
    {
        //===========matrix with border
        //offset is manage in the iterator rank
        return &data[it.getRank()];
        //=============no matrix.h
        //return subMatrix->getValuePtr(i);
    }
    //-------------------------------------------------------------------------------

    //! Get the size of matrix
    /*!
      \return the size
    */
    //-------------------------------------------------------------------------------
    unsigned int getGlobalSize()
    //-------------------------------------------------------------------------------
    {
        unsigned int res = head.width*head.height;
        return res;
    }
    //-------------------------------------------------------------------------------

    //! Get header of the matrix
    /*!
      \return the head
    */
    //-------------------------------------------------------------------------------
    HEADER getGlobalHeader()
    //-------------------------------------------------------------------------------
    {
        return head;
    }
    //-------------------------------------------------------------------------------

    //! Get header of the submatrix
    /*!
      \return the head
    */
    //-------------------------------------------------------------------------------
    HEADER getHeader()
    //-------------------------------------------------------------------------------
    {
        return loc_head;
    }
    //-------------------------------------------------------------------------------

    //! Get the size of matrix
    /*!
      \return the local size
    */
    //-------------------------------------------------------------------------------
    unsigned int getSize()
    //-------------------------------------------------------------------------------
    {
        unsigned int res = loc_head.width*loc_head.height;
        return res;
    }
    //-------------------------------------------------------------------------------

    //! Get the width of the matrix
    /*!
      \return the width
    */
    //-------------------------------------------------------------------------------
    unsigned int getWidth()
    //-------------------------------------------------------------------------------
    {
        return loc_head.width;
    }
    //-------------------------------------------------------------------------------

    //! Get the height of the matrix
    /*!
      \return the height
    */
    //-------------------------------------------------------------------------------
    unsigned int getHeight()
    //-------------------------------------------------------------------------------
    {
        return loc_head.height;
    }
    //-------------------------------------------------------------------------------

    //! Get the x of the submatrix
    /*!
      \return the x
    */
    //-------------------------------------------------------------------------------
    double getX()
    //-------------------------------------------------------------------------------
    {
        return loc_head.x;
    }
    //-------------------------------------------------------------------------------

    //! Get the y of the submatrix
    /*!
      \return the y
    */
    //-------------------------------------------------------------------------------
    double getY()
    //-------------------------------------------------------------------------------
    {
        return loc_head.y;
    }
    //-------------------------------------------------------------------------------

    //! Print the matrix by bloc
    /*!
     */
    //-------------------------------------------------------------------------------
    void print()
    //-------------------------------------------------------------------------------
    {
        std::stringstream st;
        for(unsigned int i=0; i<loc_head.height; i++) {
            for(unsigned int j=0; j<loc_head.width; j++) {
                //===========matrix with border
                //st<<data[i*border_head.width+j]<<" ";
                st<<data[border*border_head.width + border + i*border_head.width+j]<<" ";
                //=============no matrix.h
                //st<<subMatrix->getValue(i*loc_head.width+j)<<" ";
            }
            st<<"\n";
        }
        Mpiomp::print(st.str());
    }
    //-------------------------------------------------------------------------------

    //! Print the matrix by bloc
    /*!
     */
    //-------------------------------------------------------------------------------
    void printAll()
    //-------------------------------------------------------------------------------
    {
        std::stringstream st;
        st<<"\nPrint all the matrix with borders\n";
        for(unsigned int i=0; i<border_head.height; i++) {
            for(unsigned int j=0; j<border_head.width; j++) {
                st<<data[i*border_head.width+j]<<" ";
            }
            st<<"\n";
        }
        Mpiomp::print(st.str());
    }
    //-------------------------------------------------------------------------------

    //! get the 8 Neighbors of the current iterator element
    /*!
      \return returns the 8 neighbors of the current iterator element
    */
    //-------------------------------------------------------------------------------
    DMatrix<T>::neighbors get8Neighbors(iterator it,DMatrix<T>::neighbors result=NULL)
    //-------------------------------------------------------------------------------
    {
        //neighbor 0 - up left corner
        //neighbor 1 - up value
        //neighbor 2 - up right corner
        //neighbor 3 - right value
        //neighbor 4 - down right corner
        //neighbor 5 - down value
        //neighbor 6 - down left corner
        //neighbor 7 - left value

        if(NULL==result)
            result = new T[8];

        //unsigned int size = border_head.width*border_head.height;
        //===========matrix with border
        /*unsigned int size = loc_head.width*loc_head.height;
        unsigned int maxW = loc_head.width-1;
        unsigned int maxH = loc_head.height-1;
        unsigned int min = loc_head.width*maxH;
        unsigned int max = (loc_head.width*loc_head.height)-1;
        */
        // if(it.getRank()<size)
        // {
        //middle
        //===========matrix with border
        //if(itR%loc_head.width>0 && itR%loc_head.width<maxW)
        //{
        //  if(itR>maxW+1 && itR<min-1)
        //	 {
        //===========matrix with border
        //unsigned int r = itR-loc_head.width-1;
        unsigned int r = it.getRank()-border_head.width-1;
        result[0] = data[r];
        //===========matrix with border
        //r = itR-loc_head.width;
        r = it.getRank()-border_head.width;
        result[1] = data[r];
        //===========matrix with border
        //r = itR-loc_head.width+1;
        r = it.getRank()-border_head.width+1;
        result[2] = data[r];
        r = it.getRank()-1;
        result[7] = data[r];
        r = it.getRank()+1;
        result[3] = data[r];
        //===========matrix with border
        //r = itR+loc_head.width-1;
        r = it.getRank()+border_head.width-1;
        result[6] = data[r];
        //===========matrix with border
        //r = itR+loc_head.width;
        r = it.getRank()+border_head.width;
        result[5] = data[r];
        //===========matrix with border
        //r = itR+1+loc_head.width;
        r = it.getRank()+1+border_head.width;
        result[4] = data[r];
        //===========matrix with border
        /*	   }
        		 //subMatrix->getNeighborsValues(it.getRank(),result);
        //we on the up border
        else if(0<=itR && itR<=maxW)
        		 //int t=0;
        		 upNeighbors(result, it.getRank());
        //we are on the down border
        else //if(min<=itR && itR<=max)
        		 //int t=0;
        		 downNeighbors(result, it.getRank());
        	     }
        	   //we are on the left border
        	   else if(itR%loc_head.width==0)
        	     {
        //we on the up border
        if(0<=itR && itR<=maxW)
        		 leftUpNeighbors(result, it.getRank());

        //we are on the down border
        else if(min<=itR && itR<=max)
        		 leftDownNeighbors(result, it.getRank());

        else
        		 leftNeighbors(result, it.getRank());
        	     }
        	   //we are on the right border
        	   else if(itR%loc_head.width==maxW)
        	     {
        //we on the up border
        if(0<=itR && itR<=maxW)
        		 rightUpNeighbors(result, it.getRank());
        //we are on the down border
        else if(min<=itR && itR<=max)
        		 rightDownNeighbors(result, it.getRank());
        else
        		 rightNeighbors(result, it.getRank());
        		 }*/
        //  }
        return result;
    }
    //-------------------------------------------------------------------------------

    //! get the direct 4 Neighbors of the current iterator element
    /*!
      \return returns the direct 4 neighbors of the current iterator element
    */
    //-------------------------------------------------------------------------------
    DMatrix<T>::neighbors get4Neighbors(iterator it,DMatrix<T>::neighbors result)
    //-------------------------------------------------------------------------------
    {
        //neighbor 0 - up value
        //neighbor 1 - right value
        //neighbor 2 - down value
        //neighbor 3 - left value

        //if(NULL==result)
        //result = new T[4];

        unsigned int r = it.rank-border_head.width;
        result[0] = data[r];
        r = it.rank-1;
        result[3] = data[r];
        r = it.rank+1;
        result[1] = data[r];
        r = it.rank+border_head.width;
        result[2] = data[r];

        return result;
    }
    //-------------------------------------------------------------------------------

    //! get the Neighbors of the current iterator element of size border for 4 directions
    /*!
      \return returns the neighbors of the current iterator element on 4 directions for the distance border specified in the construction of the matrix
      \return size of the output table is border*4
    */
    //-------------------------------------------------------------------------------
    DMatrix<T>::neighbors getBorder4Neighbors(iterator it,DMatrix<T>::neighbors result)
    //-------------------------------------------------------------------------------
    {
        for(int i=0; i<border; i++) {
            unsigned int r = it.rank-border_head.width*(i+1);
            result[i*4] = data[r];
            r = it.rank-(i+1);
            result[3+i*4] = data[r];
            r = it.rank+(i+1);
            result[1+i*4] = data[r];
            r = it.rank+border_head.width*(i+1);
            result[2+i*4] = data[r];
        }

        return result;
    }
    //-------------------------------------------------------------------------------

    //! get the Neighbors on the X direction of the current iterator element of size border
    /*!
      \return returns the neighbors of the current iterator element on X direction for the distance border specified in the construction of the matrix
      \return size of the output table is border*2
    */
    //-------------------------------------------------------------------------------
    DMatrix<T>::neighbors getXNeighbors(iterator it,DMatrix<T>::neighbors result)
    //-------------------------------------------------------------------------------
    {
        for(int i=0; i<border; i++) {
            unsigned int r = it.rank-(i+1);
            result[i*2] = data[r];
            r = it.rank+(i+1);
            result[1+i*2] = data[r];
        }

        return result;
    }
    //-------------------------------------------------------------------------------

    //! get the Neighbors on the Y direction of the current iterator element of size border
    /*!
      \return returns the neighbors of the current iterator element on Y direction for the distance border specified in the construction of the matrix
      \return size of the output table is border*2
    */
    //-------------------------------------------------------------------------------
    DMatrix<T>::neighbors getYNeighbors(iterator it,DMatrix<T>::neighbors result)
    //-------------------------------------------------------------------------------
    {
        for(int i=0; i<border; i++) {
            unsigned int r = it.rank-border_head.width*(i+1);
            result[i*2] = data[r];
            r = it.rank+border_head.width*(i+1);
            result[1+i*2] = data[r];
        }

        return result;
    }
    //-------------------------------------------------------------------------------

    //! get the direct right Neighbors on the X direction of the current iterator element
    /*!
      \param it is the iterator of the current element
      \return returns the neighbor of the current iterator element on the right
    */
    //-------------------------------------------------------------------------------
    T getRightNeighbor(iterator it)
    //-------------------------------------------------------------------------------
    {
        unsigned r = it.rank+1;
        return data[r];
    }
    //-------------------------------------------------------------------------------

    //! get the direct left Neighbors on the X direction of the current iterator element
    /*!
      \param it is the iterator of the current element
      \return returns the neighbor of the current iterator element on the left
    */
    //-------------------------------------------------------------------------------
    T getLeftNeighbor(iterator it)
    //-------------------------------------------------------------------------------
    {
        unsigned int r = it.rank-1;
        return data[r];
    }
    //-------------------------------------------------------------------------------

    //! get the direct up Neighbors on the X direction of the current iterator element
    /*!
      \param it is the iterator of the current element
      \return returns the neighbor of the current iterator element on the up
    */
    //-------------------------------------------------------------------------------
    T getUpNeighbor(iterator it)
    //-------------------------------------------------------------------------------
    {
        unsigned int r = it.rank-border_head.width;
        return data[r];
    }
    //-------------------------------------------------------------------------------

    //! get the direct down Neighbors on the X direction of the current iterator element
    /*!
      \param it is the iterator of the current element
      \return returns the neighbor of the current iterator element on the down
    */
    //-------------------------------------------------------------------------------
    T getDownNeighbor(iterator it)
    //-------------------------------------------------------------------------------
    {
        unsigned int r = it.rank+border_head.width;
        return data[r];
    }
    //-------------------------------------------------------------------------------

    //! returns true if the current element is on the right border of the matrix
    /*!
      \param it is the iterator of the current element
      \return returns true if the current element is on the right border of the matrix
    */
    //-------------------------------------------------------------------------------
    bool isRightBorder(iterator it)
    //-------------------------------------------------------------------------------
    {
        if(it.rank%border_head.width == border_head.width-border-1)
            return true;
        else
            return false;
    }
    //-------------------------------------------------------------------------------

    //! returns true if the current element is on the down border of the matrix
    /*!
      \param it is the iterator of the current element
      \return returns true if the current element is on the down border of the matrix
    */
    //-------------------------------------------------------------------------------
    bool isDownBorder(iterator it)
    //-------------------------------------------------------------------------------
    {
        if(floor(it.rank/border_head.width) == border_head.height-border-1)
            return true;
        else
            return false;
    }
    //-------------------------------------------------------------------------------

    //! returns true if the current element is on the left border of the matrix
    /*!
      \param it is the iterator of the current element
      \return returns true if the current element is on the left border of the matrix
    */
    //-------------------------------------------------------------------------------
    bool isLeftBorder(iterator it)
    //-------------------------------------------------------------------------------
    {
        if(it.rank%border_head.width == border)
            return true;
        else
            return false;
    }
    //-------------------------------------------------------------------------------

    //! returns true if the current element is on the up border of the matrix
    /*!
      \param it is the iterator of the current element
      \return returns true if the current element is on the up border of the matrix
    */
    //-------------------------------------------------------------------------------
    bool isUpBorder(iterator it)
    //-------------------------------------------------------------------------------
    {
        if(floor(it.rank/border_head.width) == border)
            return true;
        else
            return false;
    }
    //-------------------------------------------------------------------------------

    //! returns true if the current element is on the physical border of the matrix
    /*!
      \param it is the iterator of the current element
      \return returns true if the current element is on the physical border of the matrix
    */
    //-------------------------------------------------------------------------------
    bool isPhysicalBorder(iterator it)
    //-------------------------------------------------------------------------------
    {
        if(col==0 && it.rank%border_head.width==border)
            return true;
        if(col==cols-1 && it.rank%border_head.width==border_head.width-border-1)
            return true;
        if(row==0 && floor(it.rank/border_head.width)==border)
            return true;
        if(row==rows-1 && floor(it.rank/border_head.width)==border_head.height-border-1)
            return true;
        else
            return false;
    }
    //-------------------------------------------------------------------------------


    //=======================MPI EXCHANGES METHODS=======================//

    //! get borders
    /*!
      Get the borders for submatrice (called at the beginning of the calculation).
      MPI exchanges
      \param b (border) is the size of the border
    */
    //-------------------------------------------------------------------------------
    void getBorders(/*int b*/)
    //-------------------------------------------------------------------------------
    {
        //std::cout<<"PASSAGE DANS GETBORDERS"<<std::endl;
        //for the single one subMatrice of each process
        //symetric communications between processors

        //border=b;
        if(border>0) {
            int * mpi_ranks = new int[8];
            int * omp_ranks = new int[8];
            //get the rank of the neighbours MPI processes
            get8Ranks(mpi_ranks,omp_ranks);

#ifdef DEBUG
            std::stringstream st;
            st<<"MPI Ranks 0 to 8 :\n";
            for(int i=0; i<8; i++) {
                st<<mpi_ranks[i]<<",";
            }
            st<<"\n"<<"\n";
#ifdef _OPENMP
            st<<"OMP Ranks 0 to 8 :\n";
            for(int i=0; i<8; i++) {
                st<<omp_ranks[i]<<",";
            }
            st<<"\n";
#endif
            Mpiomp::print(st.str());
#endif

            T * uplV=NULL;
            T* upV=NULL;
            T* uprV=NULL;
            T* rightV=NULL;
            T* downrV=NULL;
            T* downV=NULL;
            T* downlV=NULL;
            T* leftV=NULL;
            T * uplVR=NULL;
            T* upVR=NULL;
            T* uprVR=NULL;
            T* rightVR=NULL;
            T* downrVR=NULL;
            T* downVR=NULL;
            T* downlVR=NULL;
            T* leftVR=NULL;

            unsigned int sizeBB = border*border;
            unsigned int sizeWB = loc_head.width*border;
            unsigned int sizeHB = loc_head.height*border;
            if(mpi_ranks[0]>-1  && mpi_ranks[0]<mpi_nb && mpi_ranks[0]!=mpi_rank && border>0) {
#ifdef DEBUG
                std::stringstream st;
                st<<"Send and Rcv left up corner : MPI-"<<mpi_ranks[0]<<" OMP-"<<omp_ranks[0]<<" : \n";
                Mpiomp::print(st.str());
#endif

                uplV = getBorderToSend(col-1,row-1);
                //=============no matrix.h
                //uplV = subMatrix->getBorder(col-1,row-1,border);
                //===========matrix with border
                //uplValues = new T[sizeBB];
                uplVR = new T[sizeBB];
                BorderExchanges(uplVR, uplV, sizeBB,mpi_ranks[0], omp_ranks[0],0);
                FillBorders(0,uplVR);
                delete [] uplV;
                delete [] uplVR;
            }
            if(mpi_ranks[4]>-1 && mpi_ranks[4]<mpi_nb && mpi_ranks[4]!=mpi_rank && border>0) {
#ifdef DriEBUG
                std::stringstream st;
                st<<"Send and Rcv right down corner : MPI-"<<mpi_ranks[4]<<" OMP-"<<omp_ranks[4]<<" : \n";
                Mpiomp::print(st.str());
#endif

                downrV = getBorderToSend(col+1,row+1);
                //=============no matrix.h
                //downrV = subMatrix->getBorder(col+1,row+1,border);
                //===========matrix with border
                //downrValues = new T[sizeBB];
                downrVR = new T[sizeBB];
                BorderExchanges(downrVR, downrV, sizeBB,mpi_ranks[4], omp_ranks[4],1);
                FillBorders(4,downrVR);
                delete [] downrV;
                delete [] downrVR;
            }

            if(mpi_ranks[1]>=0 && mpi_ranks[1]<mpi_nb && mpi_ranks[1]!=mpi_rank  && border>0) {
#ifdef DEBUG
                std::stringstream st;
                st<<"Send and Rcv up values : MPI-"<<mpi_ranks[1]<<" OMP-"<<omp_ranks[1]<<" : \n";
                Mpiomp::print(st.str());
#endif

                upV = getBorderToSend(col,row-1);
                //=============no matrix.h
                //upV = subMatrix->getBorder(col,row-1,border);
                //===========matrix with border
                //upValues = new T[sizeWB];
                upVR = new T[sizeWB];
                BorderExchanges(upVR, upV, sizeWB,mpi_ranks[1], omp_ranks[1],0);
                FillBorders(1,upVR);
                delete [] upV;
                delete [] upVR;
            }
            if(mpi_ranks[5]>-1 && mpi_ranks[5]<mpi_nb && mpi_ranks[5]!=mpi_rank && border>0) {
#ifdef DEBUG
                std::stringstream st;
                st<<"Send and Rcv down values : MPI-"<<mpi_ranks[5]<<" OMP-"<<omp_ranks[5]<<" : \n";
                Mpiomp::print(st.str());
#endif

                downV = getBorderToSend(col,row+1);
                //=============no matrix.h
                //downV = subMatrix->getBorder(col,row+1,border);
                //===========matrix with border
                //downValues = new T[sizeWB];
                downVR = new T[sizeWB];
                BorderExchanges(downVR, downV, sizeWB,mpi_ranks[5], omp_ranks[5],1);
                FillBorders(5,downVR);
                delete [] downV;
                delete [] downVR;
            }

            if(mpi_ranks[2]>-1 && mpi_ranks[2]<mpi_nb && mpi_ranks[2]!=mpi_rank && border>0) {
#ifdef DEBUG
                std::stringstream st;
                st<<"Send and Rcv right up corner  : MPI-"<<mpi_ranks[2]<<" OMP-"<<omp_ranks[2]<<" : \n";
                Mpiomp::print(st.str());
#endif

                uprV = getBorderToSend(col+1,row-1);
                //=============no matrix.h
                //uprV = subMatrix->getBorder(col+1,row-1,border);
                //===========matrix with border
                //uprValues = new T[sizeBB];
                uprVR = new T[sizeBB];
                BorderExchanges(uprVR, uprV, sizeBB,mpi_ranks[2], omp_ranks[2],0);
                FillBorders(2,uprVR);
                delete [] uprV;
                delete [] uprVR;
            }
            if(mpi_ranks[6]>-1 && mpi_ranks[6]<mpi_nb && mpi_ranks[6]!=mpi_rank && border>0) {
#ifdef DEBUG
                std::stringstream st;
                st<<"Send and Rcv left down corner : MPI-"<<mpi_ranks[6]<<" OMP-"<<omp_ranks[6]<<" : \n";
                Mpiomp::print(st.str());
#endif

                downlV = getBorderToSend(col-1,row+1);
                //=============no matrix.h
                //downlV = subMatrix->getBorder(col-1,row+1,border);
                //===========matrix with border
                //downlValues = new T[sizeBB];
                downlVR = new T[sizeBB];
                BorderExchanges(downlVR, downlV, sizeBB,mpi_ranks[6], omp_ranks[6],1);
                FillBorders(6,downlVR);
                delete [] downlV;
                delete [] downlVR;
            }

            if(mpi_ranks[3]>-1 && mpi_ranks[3]<mpi_nb && mpi_ranks[3]!=mpi_rank && border>0) {
#ifdef DEBUG
                std::stringstream st;
                st<<"Send and Rcv right values : MPI-"<<mpi_ranks[3]<<" OMP-"<<omp_ranks[3]<<": \n";
                Mpiomp::print(st.str());
#endif

                rightV = getBorderToSend(col+1,row);
                //=============no matrix.h
                //rightV = subMatrix->getBorder(col+1,row,border);
                //===========matrix with border
                //rightValues = new T[sizeHB];
                rightVR = new T[sizeHB];
                BorderExchanges(rightVR, rightV, sizeHB,mpi_ranks[3], omp_ranks[3],0);
                FillBorders(3,rightVR);
                delete [] rightV;
                delete [] rightVR;
            }
            if(mpi_ranks[7]>-1 && mpi_ranks[7]<mpi_nb && mpi_ranks[7]!=mpi_rank && border>0) {
#ifdef DEBUG
                std::stringstream st;
                st<<"Send and Rcv left values : MPI-"<<mpi_ranks[7]<<" OMP-"<<omp_ranks[7]<<" : \n";
                Mpiomp::print(st.str());
#endif

                leftV = getBorderToSend(col-1,row);
                //=============no matrix.h
                //leftV = subMatrix->getBorder(col-1,row,border);
                //===========matrix with border
                //leftValues = new T[sizeHB];
                leftVR = new T[sizeHB];
                BorderExchanges(leftVR, leftV, sizeHB,mpi_ranks[7], omp_ranks[7],1);
                FillBorders(7,leftVR);
                delete [] leftV;
                delete [] leftVR;
            }

            delete [] mpi_ranks;
            mpi_ranks=NULL;
            delete [] omp_ranks;
            omp_ranks=NULL;
            /*
              #ifdef _OPENMP
              list = (Matrix<T> **)(Mpiomp::GetList(shared_mem));
              #endif
            */
        }
    }
    //-------------------------------------------------------------------------------
    //===========================================================//

};
//-------------------------------------------------------------------------------

#endif
